
import React from 'react';
import { Link } from 'react-router-dom';
import { Product } from '../types';
import { useCart } from '../context/CartContext';
import { ShoppingCart } from 'lucide-react';
import { Button, Skeleton } from './ui';

interface ProductCardProps {
  product: Product;
}

const ProductCard: React.FC<ProductCardProps> = ({ product }) => {
  const { addToCart } = useCart();

  const handleAddToCart = (e: React.MouseEvent) => {
    e.preventDefault();
    addToCart(product, 1);
  };

  return (
    <Link to={`/products/${product.id}`} className="group block">
      <div className="bg-background rounded-lg shadow-md hover:shadow-elevate transition-shadow duration-300 overflow-hidden h-full flex flex-col">
        <div className="relative overflow-hidden">
          <img
            src={product.image}
            alt={product.name}
            className="w-full h-56 object-cover group-hover:scale-110 transition-transform duration-300"
          />
        </div>
        <div className="p-4 flex flex-col flex-grow">
          <h3 className="text-lg font-bold font-heading text-brand-text-dark mb-2 truncate">{product.name}</h3>
          <p className="text-brand-text-light text-sm mb-4 flex-grow">{product.description.substring(0, 60)}...</p>
          <div className="flex items-center justify-between mt-auto">
            <span className="text-xl font-bold text-brand-purple">${product.price.toFixed(2)}</span>
            <Button
              onClick={handleAddToCart}
              size="icon"
              aria-label={`Add ${product.name} to cart`}
            >
              <ShoppingCart size={20} />
            </Button>
          </div>
        </div>
      </div>
    </Link>
  );
};

export const ProductCardSkeleton: React.FC = () => {
  return (
    <div className="bg-background rounded-lg shadow-md overflow-hidden h-full flex flex-col">
      <Skeleton className="w-full h-56" />
      <div className="p-4 flex flex-col flex-grow">
        <Skeleton className="h-5 w-3/4 mb-2" />
        <Skeleton className="h-4 w-full mb-4" />
        <div className="flex items-center justify-between mt-auto">
          <Skeleton className="h-6 w-1/4" />
          <Skeleton className="h-10 w-10" />
        </div>
      </div>
    </div>
  );
};

export default ProductCard;