



import React, { createContext, useContext, forwardRef, useState } from 'react';
import { X } from 'lucide-react';

// --- Skeleton Component ---
export const Skeleton = ({ className, ...props }: React.HTMLAttributes<HTMLDivElement>) => (
    <div
      className={`animate-pulse rounded-md bg-muted ${className}`}
      {...props}
    />
);
Skeleton.displayName = "Skeleton";

// --- Helper for class variance ---
// Simplified CVA-like function
// FIX: Added types for cva function arguments.
const cva = (base: string, { variants, defaultVariants }: { variants: any, defaultVariants: any }) => {
  return (props: any) => {
    const variantClasses = Object.keys(variants).map(key => {
      const propValue = props?.[key] || defaultVariants?.[key];
      return variants[key][propValue] || '';
    }).join(' ');
    return [base, variantClasses, props?.className].filter(Boolean).join(' ');
  }
}

// --- Button Component ---
const buttonVariants = cva(
  'inline-flex items-center justify-center rounded-md text-sm font-medium ring-offset-background transition-colors focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 disabled:pointer-events-none disabled:opacity-50',
  {
    variants: {
      variant: {
        default: 'bg-primary text-primary-foreground hover:bg-primary/90',
        hero: 'bg-gradient-hero text-white font-bold hover:opacity-90 transition-opacity',
        destructive: 'bg-red-500 text-white hover:bg-red-500/90',
        outline: 'border border-input bg-background hover:bg-gray-100 hover:text-gray-900',
        secondary: 'bg-secondary text-secondary-foreground hover:bg-secondary/80',
        ghost: 'hover:bg-gray-100 hover:text-gray-900',
        link: 'text-primary underline-offset-4 hover:underline',
      },
      size: {
        default: 'h-10 px-4 py-2',
        sm: 'h-9 rounded-md px-3',
        lg: 'h-11 rounded-md px-8',
        icon: 'h-10 w-10',
      },
    },
    defaultVariants: {
      variant: 'default',
      size: 'default',
    },
  }
);

// FIX: Added props interface for Button component to accept all standard button attributes and children.
export interface ButtonProps extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  variant?: 'default' | 'hero' | 'destructive' | 'outline' | 'secondary' | 'ghost' | 'link';
  size?: 'default' | 'sm' | 'lg' | 'icon';
  asChild?: boolean;
}

export const Button = forwardRef<HTMLButtonElement, ButtonProps>(({ className, variant, size, asChild = false, ...props }, ref) => {
  const Comp = asChild ? "span" : 'button';
  return (
    <Comp
      className={buttonVariants({ variant, size, className })}
      ref={ref as any}
      {...props}
    />
  );
});
Button.displayName = "Button";

// --- Card Component ---
// FIX: Added explicit types for Card and its sub-components to accept children and other HTML attributes.
export const Card = forwardRef<HTMLDivElement, React.HTMLAttributes<HTMLDivElement>>(({ className, ...props }, ref) => (
  <div
    ref={ref}
    className={`rounded-lg border bg-card text-card-foreground shadow-sm ${className}`}
    {...props}
  />
));
Card.displayName = "Card";

export const CardHeader = forwardRef<HTMLDivElement, React.HTMLAttributes<HTMLDivElement>>(({ className, ...props }, ref) => (
  <div ref={ref} className={`flex flex-col space-y-1.5 p-6 ${className}`} {...props} />
));
CardHeader.displayName = "CardHeader";

export const CardTitle = forwardRef<HTMLHeadingElement, React.HTMLAttributes<HTMLHeadingElement>>(({ className, ...props }, ref) => (
  <h3 ref={ref} className={`text-2xl font-semibold leading-none tracking-tight font-heading ${className}`} {...props} />
));
CardTitle.displayName = "CardTitle";

export const CardDescription = forwardRef<HTMLParagraphElement, React.HTMLAttributes<HTMLParagraphElement>>(({ className, ...props }, ref) => (
  <p ref={ref} className={`text-sm text-muted-foreground ${className}`} {...props} />
));
CardDescription.displayName = "CardDescription";

export const CardContent = forwardRef<HTMLDivElement, React.HTMLAttributes<HTMLDivElement>>(({ className, ...props }, ref) => (
  <div ref={ref} className={`p-6 pt-0 ${className}`} {...props} />
));
CardContent.displayName = "CardContent";

export const CardFooter = forwardRef<HTMLDivElement, React.HTMLAttributes<HTMLDivElement>>(({ className, ...props }, ref) => (
  <div ref={ref} className={`flex items-center p-6 pt-0 ${className}`} {...props} />
));
CardFooter.displayName = "CardFooter";

// --- Input Component ---
// FIX: Added props interface for Input component to accept all standard input attributes.
export interface InputProps extends React.InputHTMLAttributes<HTMLInputElement> {}
export const Input = forwardRef<HTMLInputElement, InputProps>(({ className, type, ...props }, ref) => {
  return (
    <input
      type={type}
      className={`flex h-10 w-full rounded-md border border-gray-300 bg-background px-3 py-2 text-sm ring-offset-background file:border-0 file:bg-transparent file:text-sm file:font-medium placeholder:text-gray-500 focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-brand-purple focus-visible:ring-offset-2 disabled:cursor-not-allowed disabled:opacity-50 ${className}`}
      ref={ref}
      {...props}
    />
  );
});
Input.displayName = "Input";

// --- Label Component ---
// FIX: Added explicit types for Label component.
export const Label = forwardRef<HTMLLabelElement, React.LabelHTMLAttributes<HTMLLabelElement>>(({ className, ...props }, ref) => (
    <label ref={ref} className={`text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70 ${className}`} {...props} />
));
Label.displayName = "Label";

// --- Textarea Component ---
// FIX: Added props interface for Textarea component to accept all standard textarea attributes.
export interface TextareaProps extends React.TextareaHTMLAttributes<HTMLTextAreaElement> {}
export const Textarea = forwardRef<HTMLTextAreaElement, TextareaProps>(({ className, ...props }, ref) => {
  return (
    <textarea
      className={`flex min-h-[80px] w-full rounded-md border border-gray-300 bg-background px-3 py-2 text-sm ring-offset-background placeholder:text-gray-500 focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-brand-purple focus-visible:ring-offset-2 disabled:cursor-not-allowed disabled:opacity-50 ${className}`}
      ref={ref}
      {...props}
    />
  );
});
Textarea.displayName = "Textarea";

// --- Tabs Component ---
// FIX: Implemented a functional Tabs component with context and state to fix type errors and enable functionality.
interface TabsContextType {
  activeValue: string;
  setActiveValue: (value: string) => void;
}
const TabsContext = createContext<TabsContextType | undefined>(undefined);

export const Tabs = ({ className, defaultValue, children, ...props }: React.HTMLAttributes<HTMLDivElement> & { defaultValue: string }) => {
  const [activeValue, setActiveValue] = useState(defaultValue);
  return (
    <TabsContext.Provider value={{ activeValue, setActiveValue }}>
      <div className={className} {...props}>{children}</div>
    </TabsContext.Provider>
  );
};

export const TabsList = forwardRef<HTMLDivElement, React.HTMLAttributes<HTMLDivElement>>(({ className, ...props }, ref) => (
  <div
    ref={ref}
    className={`inline-flex h-10 items-center justify-center rounded-md bg-muted p-1 text-muted-foreground ${className}`}
    {...props}
  />
));
TabsList.displayName = "TabsList";

interface TabsTriggerProps extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  value: string;
}
export const TabsTrigger = forwardRef<HTMLButtonElement, TabsTriggerProps>(({ className, value, children, ...props }, ref) => {
  const context = useContext(TabsContext);
  if (!context) {
    throw new Error('TabsTrigger must be used within a Tabs component');
  }
  const isActive = context.activeValue === value;
  return (
    <button
      ref={ref}
      onClick={() => context.setActiveValue(value)}
      data-state={isActive ? 'active' : 'inactive'}
      className={`inline-flex items-center justify-center whitespace-nowrap rounded-sm px-3 py-1.5 text-sm font-medium ring-offset-background transition-all focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 disabled:pointer-events-none disabled:opacity-50 data-[state=active]:bg-background data-[state=active]:text-foreground data-[state=active]:shadow-sm ${className}`}
      {...props}
    >
      {children}
    </button>
  );
});
TabsTrigger.displayName = "TabsTrigger";

interface TabsContentProps extends React.HTMLAttributes<HTMLDivElement> {
  value: string;
}
export const TabsContent = forwardRef<HTMLDivElement, TabsContentProps>(({ className, value, children, ...props }, ref) => {
  const context = useContext(TabsContext);
  if (!context) {
    throw new Error('TabsContent must be used within a Tabs component');
  }
  const isVisible = context.activeValue === value;
  return (
    <div
      ref={ref}
      style={{ display: isVisible ? 'block' : 'none' }}
      className={`mt-2 ring-offset-background focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 ${className}`}
      {...props}
    >
      {children}
    </div>
  );
});
TabsContent.displayName = "TabsContent";

// --- Dialog Component ---
// FIX: Added explicit types for Dialog context and components.
interface DialogContextType {
  open: boolean;
  onOpenChange: (open: boolean) => void;
}
const DialogContext = createContext<DialogContextType | undefined>(undefined);

export const Dialog = ({ open, onOpenChange, children }: React.PropsWithChildren<{ open: boolean, onOpenChange: (open: boolean) => void }>) => {
    return <DialogContext.Provider value={{ open, onOpenChange }}>{children}</DialogContext.Provider>;
};

// FIX: Implement `asChild` prop for DialogTrigger to fix type error and avoid wrapping.
// FIX: Changed props to use React.PropsWithChildren to resolve issue with children prop type inference when using `asChild`.
export const DialogTrigger = ({ children, asChild = false }: React.PropsWithChildren<{ asChild?: boolean }>) => {
    const context = useContext(DialogContext);
    if (!context) {
      throw new Error("DialogTrigger must be used within a Dialog");
    }

    if (asChild) {
        if (React.isValidElement(children)) {
            const child = React.Children.only(children);
            // FIX: Assert child.props to be an object with an optional onClick handler to resolve spread and property access errors.
            const childProps = child.props as { onClick?: (...args: any[]) => void; [key: string]: any };

            // FIX: Add generic type <any> to React.cloneElement to resolve overload error.
            return React.cloneElement<any>(child, {
                ...childProps,
                onClick: (...args: any[]) => {
                    if (childProps.onClick) {
                        childProps.onClick(...args);
                    }
                    context.onOpenChange(true);
                }
            });
        }
    }
    return <div onClick={() => context.onOpenChange(true)}>{children}</div>;
};

// FIX: Added explicit props type to make className optional.
export const DialogContent = forwardRef<HTMLDivElement, React.HTMLAttributes<HTMLDivElement>>(({ className, children, ...props }, ref) => {
    const context = useContext(DialogContext);

    if (!context || !context.open) return null;

    return (
        <div 
          ref={ref}
          className="fixed inset-0 z-50 bg-black/60 flex items-center justify-center"
          onClick={() => context.onOpenChange(false)}
        >
            <div 
              className={`relative bg-white rounded-lg shadow-xl p-6 w-full max-w-lg ${className}`}
              onClick={(e) => e.stopPropagation()}
              {...props}
            >
                {children}
                <button onClick={() => context.onOpenChange(false)} className="absolute top-4 right-4 text-gray-400 hover:text-gray-600">
                    <X size={24} />
                </button>
            </div>
        </div>
    );
});
DialogContent.displayName = "DialogContent";

// FIX: Added explicit types for Dialog sub-components.
export const DialogHeader = ({ className, ...props }: React.HTMLAttributes<HTMLDivElement>) => (
    <div className={`flex flex-col space-y-1.5 text-center sm:text-left ${className}`} {...props} />
);
export const DialogTitle = forwardRef<HTMLHeadingElement, React.HTMLAttributes<HTMLHeadingElement>>(({ className, ...props }, ref) => (
    <h2 ref={ref} className={`text-lg font-semibold leading-none tracking-tight font-heading ${className}`} {...props} />
));
DialogTitle.displayName = "DialogTitle";
export const DialogDescription = forwardRef<HTMLParagraphElement, React.HTMLAttributes<HTMLParagraphElement>>(({ className, ...props }, ref) => (
    <p ref={ref} className={`text-sm text-muted-foreground ${className}`} {...props} />
));
DialogDescription.displayName = "DialogDescription";
export const DialogFooter = ({ className, ...props }: React.HTMLAttributes<HTMLDivElement>) => (
    <div className={`flex flex-col-reverse sm:flex-row sm:justify-end sm:space-x-2 mt-6 ${className}`} {...props} />
);

// --- Badge Component ---
const badgeVariants = cva(
  "inline-flex items-center rounded-full border px-2.5 py-0.5 text-xs font-semibold transition-colors focus:outline-none focus:ring-2 focus:ring-ring focus:ring-offset-2",
  {
    variants: {
      variant: {
        default: "border-transparent bg-primary text-primary-foreground",
        secondary: "border-transparent bg-secondary text-secondary-foreground",
        destructive: "border-transparent bg-red-100 text-red-800",
        outline: "text-foreground",
        success: "border-transparent bg-green-100 text-green-800",
        warning: "border-transparent bg-yellow-100 text-yellow-800",
      },
    },
    defaultVariants: {
      variant: "default",
    },
  }
)
// FIX: Added explicit props type to make className and variant optional.
interface BadgeProps extends React.HTMLAttributes<HTMLDivElement> {
  variant?: 'default' | 'secondary' | 'destructive' | 'outline' | 'success' | 'warning';
}
// FIX: Refactored Badge to use forwardRef for consistency and to resolve a type error where 'className' was not being correctly inferred on BadgeProps.
export const Badge = forwardRef<HTMLDivElement, BadgeProps>(({ className, variant, ...props }, ref) => {
  return (<div ref={ref} className={badgeVariants({ variant, className })} {...props} />);
});
Badge.displayName = "Badge";


// --- Select Component ---
// FIX: Added explicit types for Select and SelectValue components.
export const Select = forwardRef<HTMLSelectElement, React.SelectHTMLAttributes<HTMLSelectElement>>(({ className, children, ...props }, ref) => (
    <select
        className={`flex h-10 w-full items-center justify-between rounded-md border border-gray-300 bg-background px-3 py-2 text-sm ring-offset-background placeholder:text-muted-foreground focus:outline-none focus:ring-2 focus:ring-brand-purple focus:ring-offset-2 disabled:cursor-not-allowed disabled:opacity-50 ${className}`}
        ref={ref}
        {...props}
    >
        {children}
    </select>
));
Select.displayName = "Select";

// FIX: Updated SelectValue to accept all standard option attributes to fix type errors when passing the 'value' prop.
export const SelectValue = (props: React.OptionHTMLAttributes<HTMLOptionElement>) => <option {...props} />;
SelectValue.displayName = "SelectValue";