import React, { createContext, useContext, useState, useEffect, useCallback } from 'react';
import { DEFAULT_LOGO_BASE64 } from '../constants';

interface BrandingContextType {
  logo: string;
  sliderImages: string[];
  updateLogo: (newLogo: string) => void;
  updateSliderImages: (images: string[]) => void;
  resetBranding: () => void;
}

const BrandingContext = createContext<BrandingContextType | undefined>(undefined);

export const useBranding = () => {
  const context = useContext(BrandingContext);
  if (context === undefined) {
    throw new Error('useBranding must be used within a BrandingProvider');
  }
  return context;
};

const LOGO_KEY = 'lanisdeb-logo';
const SLIDER_KEY = 'lanisdeb-slider-images';

// FIX: Changed props to use React.PropsWithChildren for consistency with other providers.
export const BrandingProvider = ({ children }: React.PropsWithChildren<{}>) => {
  const [logo, setLogo] = useState(DEFAULT_LOGO_BASE64);
  const [sliderImages, setSliderImages] = useState<string[]>([]);

  useEffect(() => {
    try {
      const storedLogo = localStorage.getItem(LOGO_KEY);
      if (storedLogo) {
        setLogo(storedLogo);
      }
      const storedSlider = localStorage.getItem(SLIDER_KEY);
      if (storedSlider) {
        setSliderImages(JSON.parse(storedSlider));
      }
    } catch (error) {
      console.error("Failed to load branding from localStorage", error);
    }
  }, []);

  useEffect(() => {
    const link = document.querySelector<HTMLLinkElement>("link[rel~='icon']");
    if (link) {
      link.href = logo;
    }
  }, [logo]);

  const updateLogo = useCallback((newLogo: string) => {
    setLogo(newLogo);
    localStorage.setItem(LOGO_KEY, newLogo);
  }, []);

  const updateSliderImages = useCallback((images: string[]) => {
    setSliderImages(images);
    localStorage.setItem(SLIDER_KEY, JSON.stringify(images));
  }, []);
  
  const resetBranding = useCallback(() => {
     localStorage.removeItem(LOGO_KEY);
     localStorage.removeItem(SLIDER_KEY);
     setLogo(DEFAULT_LOGO_BASE64);
     setSliderImages([]);
  }, []);


  const value = { logo, sliderImages, updateLogo, updateSliderImages, resetBranding };

  return <BrandingContext.Provider value={value}>{children}</BrandingContext.Provider>;
};