import React, { createContext, useContext, useState, useEffect, ReactNode, useCallback } from 'react';
import { CartItem, Product } from '../types';

interface ToastMessage {
  id: number;
  message: string;
}

interface CartContextType {
  cartItems: CartItem[];
  addToCart: (product: Product, quantity: number) => void;
  removeFromCart: (productId: string) => void;
  updateQuantity: (productId: string, newQuantity: number) => void;
  clearCart: () => void;
  cartCount: number;
  cartTotal: number;
  toastMessages: ToastMessage[];
  showToast: (message: string) => void;
}

const CartContext = createContext<CartContextType | undefined>(undefined);

export const useCart = () => {
  const context = useContext(CartContext);
  if (context === undefined) {
    throw new Error('useCart must be used within a CartProvider');
  }
  return context;
};

// FIX: Changed props to use React.PropsWithChildren to resolve issue with children prop type.
export const CartProvider = ({ children }: React.PropsWithChildren<{}>) => {
  const [cartItems, setCartItems] = useState<CartItem[]>([]);
  const [toastMessages, setToastMessages] = useState<ToastMessage[]>([]);

  useEffect(() => {
    try {
      const storedCart = localStorage.getItem('lanisdeb-cart');
      if (storedCart) {
        setCartItems(JSON.parse(storedCart));
      }
    } catch (error) {
      console.error("Failed to parse cart from localStorage", error);
      localStorage.removeItem('lanisdeb-cart');
    }
  }, []);

  useEffect(() => {
    localStorage.setItem('lanisdeb-cart', JSON.stringify(cartItems));
  }, [cartItems]);

  const showToast = useCallback((message: string) => {
    const id = Date.now();
    setToastMessages(prev => [...prev, { id, message }]);
    setTimeout(() => {
      setToastMessages(prev => prev.filter(toast => toast.id !== id));
    }, 3000);
  }, []);

  const addToCart = (product: Product, quantity: number) => {
    setCartItems(prevItems => {
      const existingItem = prevItems.find(item => item.id === product.id);
      if (existingItem) {
        const newQuantity = Math.min(existingItem.quantity + quantity, product.stock);
        return prevItems.map(item =>
          item.id === product.id ? { ...item, quantity: newQuantity } : item
        );
      } else {
        return [...prevItems, { ...product, quantity }];
      }
    });
    showToast(`${product.name} added to cart!`);
  };

  const removeFromCart = (productId: string) => {
    setCartItems(prevItems => prevItems.filter(item => item.id !== productId));
    showToast(`Item removed from cart.`);
  };

  const updateQuantity = (productId: string, newQuantity: number) => {
    if (newQuantity <= 0) {
      removeFromCart(productId);
    } else {
      setCartItems(prevItems =>
        prevItems.map(item =>
          item.id === productId
            ? { ...item, quantity: Math.min(newQuantity, item.stock) }
            : item
        )
      );
    }
  };

  const clearCart = () => {
    setCartItems([]);
  };

  const cartCount = cartItems.reduce((acc, item) => acc + item.quantity, 0);
  const cartTotal = cartItems.reduce((acc, item) => acc + item.price * item.quantity, 0);

  const value = {
    cartItems,
    addToCart,
    removeFromCart,
    updateQuantity,
    clearCart,
    cartCount,
    cartTotal,
    toastMessages,
    showToast
  };

  return <CartContext.Provider value={value}>{children}</CartContext.Provider>;
};