

import { Product, Category, Order, OrderStatus } from '../types';

// Mock Data
const defaultCategories: Category[] = [
  { id: 'cat1', name: 'Flour & Grains', image: 'https://images.unsplash.com/photo-1509358271058-d3433565ac22?q=80&w=1200&auto=format&fit=crop' },
  { id: 'cat2', name: 'Spices & Seasoning', image: 'https://images.unsplash.com/photo-1476533994320-56f543e4e132?q=80&w=1200&auto=format&fit=crop' },
  { id: 'cat3', name: 'Oils & Sauces', image: 'https://images.unsplash.com/photo-1572483860010-86f375f64390?q=80&w=1200&auto=format&fit=crop' },
  { id: 'cat4', name: 'Snacks', image: 'https://images.unsplash.com/photo-1599412257635-a131a18a9942?q=80&w=1200&auto=format&fit=crop' },
  { id: 'cat5', name: 'Drinks', image: 'https://images.unsplash.com/photo-1551024709-8f237c20454d?q=80&w=1200&auto=format&fit=crop' },
  { id: 'cat6', name: 'Feminine & Cosmetics' },
  { id: 'cat7', name: 'Gent items' },
  { id: 'cat8', name: 'Aso oke/Gele/Turbans' },
  { id: 'cat9', name: 'Ready to Wear Women' },
  { id: 'cat10', name: 'Ready to Wear Men' },
  { id: 'cat11', name: 'Skin Care' },
  { id: 'cat12', name: 'Spiritual items' },
  { id: 'cat13', name: 'Cloths' },
  { id: 'cat14', name: 'Food items' },
  { id: 'cat15', name: 'Groceries' },
];

const initialProducts: Product[] = [
  {
    id: 'prod1',
    name: 'Poundo Iyan (Yam Flour)',
    price: 10.99,
    description: 'Premium quality yam flour, perfect for making smooth and delicious pounded yam.',
    image: 'https://images.unsplash.com/photo-1593392945731-507a7e9f3b1e?q=80&w=800&auto=format&fit=crop',
    category: 'Flour & Grains',
    stock: 50,
    featured: true,
  },
  {
    id: 'prod2',
    name: 'Jollof Rice Spice Mix',
    price: 4.50,
    description: 'An authentic blend of spices to create the perfect Nigerian Jollof rice every time.',
    image: 'https://images.unsplash.com/photo-1589301760014-d929f39791e8?q=80&w=800&auto=format&fit=crop',
    category: 'Spices & Seasoning',
    stock: 120,
    featured: true,
  },
  {
    id: 'prod3',
    name: 'Chin Chin Snack',
    price: 3.99,
    description: 'Crunchy, sweet, and addictive Nigerian pastry snack. Perfect for any occasion.',
    image: 'https://images.unsplash.com/photo-1620800340159-913a968a3c53?q=80&w=800&auto=format&fit=crop',
    category: 'Snacks',
    stock: 80,
  },
  {
    id: 'prod4',
    name: 'Red Palm Oil',
    price: 8.75,
    description: '100% pure and natural red palm oil, essential for traditional African cuisine.',
    image: 'https://images.unsplash.com/photo-1615386095328-3615e4559816?q=80&w=800&auto=format&fit=crop',
    category: 'Oils & Sauces',
    stock: 40,
    featured: true,
  },
  {
    id: 'prod5',
    name: 'Hibiscus Flower (Zobo)',
    price: 5.25,
    description: 'Dried hibiscus flowers for making the popular refreshing Zobo drink.',
    image: 'https://images.unsplash.com/photo-1578916399995-ef0f789e9f64?q=80&w=800&auto=format&fit=crop',
    category: 'Drinks',
    stock: 60,
  },
  {
    id: 'prod6',
    name: 'Gari (Cassava Flakes)',
    price: 7.99,
    description: 'Fine-grain processed cassava, a staple food that can be soaked or used for Eba.',
    image: 'https://images.unsplash.com/photo-1518791841217-8f162f1e1131?q=80&w=800&auto=format&fit=crop',
    category: 'Flour & Grains',
    stock: 100,
  },
  {
    id: 'prod7',
    name: 'Suya Spice (Yaji)',
    price: 6.00,
    description: 'A hot and spicy mix for grilling meat, famously used for Nigerian Suya skewers.',
    image: 'https://images.unsplash.com/photo-1599421498111-ad70b472c694?q=80&w=800&auto=format&fit=crop',
    category: 'Spices & Seasoning',
    stock: 90,
    featured: true,
  },
  {
    id: 'prod8',
    name: 'Plantain Chips',
    price: 2.50,
    description: 'Crispy and savory fried plantain chips. A classic West African snack.',
    image: 'https://images.unsplash.com/photo-1600735709550-20357835f874?q=80&w=800&auto=format&fit=crop',
    category: 'Snacks',
    stock: 150,
  },
];


// Helper functions for localStorage
const getFromStorage = <T>(key: string, defaultValue: T): T => {
  try {
    const item = localStorage.getItem(key);
    if (item === null) {
      saveToStorage(key, defaultValue);
      return defaultValue;
    }
    return JSON.parse(item);
  } catch (error) {
    console.error(`Error reading from localStorage key “${key}”:`, error);
    return defaultValue;
  }
};

const saveToStorage = <T>(key: string, value: T) => {
  try {
    localStorage.setItem(key, JSON.stringify(value));
  } catch (error) {
    console.error(`Error writing to localStorage key “${key}”:`, error);
  }
};

// Initialize data if not present
getFromStorage('lanisdeb-products', initialProducts);
getFromStorage('lanisdeb-categories', defaultCategories);
getFromStorage('lanisdeb-orders', []);

// Function to generate unique IDs
const generateId = () => `id_${new Date().getTime()}_${Math.random().toString(36).substr(2, 9)}`;

// Product Functions
export const getProducts = async (): Promise<Product[]> => {
  return [...getFromStorage('lanisdeb-products', [])];
};

export const getProductById = async (id: string): Promise<Product | undefined> => {
  const currentProducts = getFromStorage('lanisdeb-products', []);
  return currentProducts.find(p => p.id === id);
};

export const getFeaturedProducts = async (): Promise<Product[]> => {
  const currentProducts = getFromStorage('lanisdeb-products', []);
  return currentProducts.filter(p => p.featured);
};

export const addProduct = async (productData: Omit<Product, 'id'>): Promise<Product> => {
  let currentProducts = getFromStorage('lanisdeb-products', []);
  if (currentProducts.some(p => p.name.trim().toLowerCase() === productData.name.trim().toLowerCase())) {
    throw new Error(`Product with name "${productData.name}" already exists.`);
  }
  const newProduct: Product = { ...productData, id: generateId() };
  currentProducts = [...currentProducts, newProduct];
  saveToStorage('lanisdeb-products', currentProducts);
  return newProduct;
};

export const updateProduct = async (updatedProduct: Product): Promise<Product> => {
  let currentProducts = getFromStorage('lanisdeb-products', []);
  if (currentProducts.some(p => p.id !== updatedProduct.id && p.name.trim().toLowerCase() === updatedProduct.name.trim().toLowerCase())) {
      throw new Error(`Product with name "${updatedProduct.name}" already exists.`);
  }
  currentProducts = currentProducts.map(p => p.id === updatedProduct.id ? updatedProduct : p);
  saveToStorage('lanisdeb-products', currentProducts);
  return updatedProduct;
};

export const deleteProduct = async (productId: string): Promise<void> => {
  let currentProducts = getFromStorage('lanisdeb-products', []);
  currentProducts = currentProducts.filter(p => p.id !== productId);
  saveToStorage('lanisdeb-products', currentProducts);
};

// Category Functions
export const getCategories = async (): Promise<Category[]> => {
    const currentProducts = getFromStorage('lanisdeb-products', []);
    const currentCategories = getFromStorage('lanisdeb-categories', []);
    return [...currentCategories].map(cat => ({
        ...cat,
        productCount: currentProducts.filter(p => p.category === cat.name).length
    })).sort((a,b) => a.name.localeCompare(b.name));
};

export const addCategory = async (categoryData: Omit<Category, 'id' | 'productCount'>): Promise<Category> => {
    let currentCategories = getFromStorage('lanisdeb-categories', []);
    const existing = currentCategories.find(c => c.name.trim().toLowerCase() === categoryData.name.trim().toLowerCase());
    if (existing) {
        // Update existing category image if provided
        existing.image = categoryData.image || existing.image;
        saveToStorage('lanisdeb-categories', currentCategories);
        return existing;
    }
    
    const newCategory: Category = { ...categoryData, name: categoryData.name.trim(), id: generateId() };
    currentCategories = [...currentCategories, newCategory];
    saveToStorage('lanisdeb-categories', currentCategories);
    return newCategory;
}

export const deleteCategory = async (categoryId: string): Promise<void> => {
    let currentCategories = getFromStorage('lanisdeb-categories', []);
    currentCategories = currentCategories.filter(c => c.id !== categoryId);
    saveToStorage('lanisdeb-categories', currentCategories);
}

// Order Functions
export const getOrders = async (): Promise<Order[]> => {
  const currentOrders = getFromStorage('lanisdeb-orders', []);
  return [...currentOrders].sort((a, b) => new Date(b.date).getTime() - new Date(a.date).getTime());
};

export const addOrder = async (orderDetails: Omit<Order, 'id' | 'date'>): Promise<Order> => {
  let currentOrders = getFromStorage('lanisdeb-orders', []);
  const newOrder: Order = {
    ...orderDetails,
    id: generateId().substring(0, 8),
    date: new Date().toISOString(),
  };
  currentOrders = [newOrder, ...currentOrders];
  saveToStorage('lanisdeb-orders', currentOrders);
  return newOrder;
};

export const updateOrderStatus = async (orderId: string, status: OrderStatus): Promise<Order | undefined> => {
  let currentOrders = getFromStorage('lanisdeb-orders', []);
  let updatedOrder: Order | undefined;
  currentOrders = currentOrders.map(o => {
    if (o.id === orderId) {
      updatedOrder = { ...o, status };
      return updatedOrder;
    }
    return o;
  });
  if (updatedOrder) {
    saveToStorage('lanisdeb-orders', currentOrders);
  }
  return updatedOrder;
};