import React, { useState } from 'react';
import { useAuth } from '../context/AuthContext';
import { useBranding } from '../context/BrandingContext';
import { Navigate, Link } from 'react-router-dom';
import { Button, Input, Card, CardHeader, CardTitle, CardContent, CardFooter } from '../components/ui';

const AdminLoginPage: React.FC = () => {
    const [password, setPassword] = useState('');
    const [error, setError] = useState('');
    const [isLoading, setIsLoading] = useState(false);
    const auth = useAuth();
    const { logo } = useBranding();

    if (auth.isLoggedIn) {
        return <Navigate to="/admin" replace />;
    }

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setError('');
        setIsLoading(true);

        // Simulate network delay for better UX
        await new Promise(res => setTimeout(res, 500));
        
        const success = auth.login(password);
        if (!success) {
            setError('Invalid password. Please try again.');
        }
        setIsLoading(false);
    };

    return (
        <div className="min-h-screen flex items-center justify-center bg-brand-bg-light p-4">
            <Card className="w-full max-w-sm">
                <CardHeader className="text-center">
                    <img src={logo} alt="Lanisdeb Market Logo" className="h-20 mx-auto mb-4" />
                    <CardTitle>Admin Login</CardTitle>
                </CardHeader>
                <form onSubmit={handleSubmit}>
                    <CardContent className="space-y-4">
                        <div className="space-y-1">
                            <label htmlFor="password">Password</label>
                            <Input
                                id="password"
                                type="password"
                                value={password}
                                onChange={(e) => setPassword(e.target.value)}
                                placeholder="Enter your password"
                                required
                                disabled={isLoading}
                            />
                        </div>
                        {error && <p className="text-sm text-red-500">{error}</p>}
                    </CardContent>
                    <CardFooter>
                        <div className="w-full flex flex-col items-center gap-2">
                            <Button type="submit" className="w-full" disabled={isLoading}>
                                {isLoading ? 'Logging in...' : 'Login'}
                            </Button>
                            <Button asChild variant="link" className="text-sm font-normal text-brand-text-light hover:text-brand-text-dark">
                                <Link to="/">Authorized Staff Only! Go Back</Link>
                            </Button>
                        </div>
                    </CardFooter>
                </form>
            </Card>
        </div>
    );
};

export default AdminLoginPage;