import React, { useState, useEffect, useCallback, useMemo } from 'react';
import { Product, Order, OrderStatus, Category } from '../types';
import { getProducts, getOrders, addProduct, updateProduct, deleteProduct, updateOrderStatus, getCategories, addCategory, deleteCategory } from '../data/db';
import { useAuth } from '../context/AuthContext';
import { useBranding } from '../context/BrandingContext';
import { Button, Input, Textarea, Badge, Tabs, TabsList, TabsTrigger, TabsContent, Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter, Select } from '../components/ui';
import { Edit, Trash2, PlusCircle, LogOut, UploadCloud, ChevronDown, ChevronUp, X, RotateCcw } from 'lucide-react';
import useDocumentTitle from '../hooks/useDocumentTitle';

// Reusable Product Edit Dialog
interface ProductEditDialogProps {
    open: boolean;
    onOpenChange: (open: boolean) => void;
    product: Partial<Product> | null;
    onSave: (product: Product) => Promise<void>;
    categories: Category[];
}

const ProductEditDialog: React.FC<ProductEditDialogProps> = ({ open, onOpenChange, product, onSave, categories }) => {
    const [editingProduct, setEditingProduct] = useState<Partial<Product> | null>(product);

    useEffect(() => {
        setEditingProduct(product);
    }, [product]);

    const handleFieldChange = (field: keyof Omit<Product, 'id'>, value: any) => {
        if (editingProduct) {
            setEditingProduct({ ...editingProduct, [field]: value });
        }
    };
    
    const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0];
        if (file) {
            const reader = new FileReader();
            reader.onloadend = () => {
                handleFieldChange('image', reader.result as string);
            };
            reader.readAsDataURL(file);
        }
    };


    const handleSave = () => {
        if (!editingProduct || !editingProduct.name || !editingProduct.category) {
            alert("Please fill in at least name and category.");
            return;
        }
        
        const productData = {
            ...editingProduct,
            price: Number(editingProduct.price) || 0,
            stock: Number(editingProduct.stock) || 0,
        } as Product;
        onSave(productData);
    };

    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            <DialogContent>
                <DialogHeader>
                    <DialogTitle>{editingProduct?.id ? 'Edit Product' : 'Add New Product'}</DialogTitle>
                    <DialogDescription>Fill in the details for the product.</DialogDescription>
                </DialogHeader>
                <div className="grid gap-4 py-4 max-h-[60vh] overflow-y-auto pr-4">
                    <Input placeholder="Name" value={editingProduct?.name || ''} onChange={(e) => handleFieldChange('name', e.target.value)} />
                    <Textarea placeholder="Description" value={editingProduct?.description || ''} onChange={(e) => handleFieldChange('description', e.target.value)} />
                    <Input placeholder="Price" type="number" value={editingProduct?.price || ''} onChange={(e) => handleFieldChange('price', e.target.value)} />
                    <Input placeholder="Stock" type="number" value={editingProduct?.stock || ''} onChange={(e) => handleFieldChange('stock', e.target.value)} />
                     <Select value={editingProduct?.category || ''} onChange={(e) => handleFieldChange('category', e.target.value)}>
                        <option value="" disabled>Select a category</option>
                        {categories.map(cat => <option key={cat.id} value={cat.name}>{cat.name}</option>)}
                    </Select>
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">Product Image</label>
                        {editingProduct?.image && <img src={editingProduct.image} alt="Preview" className="w-20 h-20 object-cover rounded mb-2"/>}
                        <Input type="file" accept="image/*" onChange={handleImageChange} />
                    </div>
                    <div className="flex items-center space-x-2">
                        <input type="checkbox" id="featured" checked={editingProduct?.featured || false} onChange={(e) => handleFieldChange('featured', e.target.checked)} className="h-4 w-4 rounded border-gray-300 text-brand-purple focus:ring-brand-purple"/>
                        <label htmlFor="featured" className="text-sm font-medium text-gray-700">Featured Product</label>
                    </div>
                </div>
                <DialogFooter>
                    <Button variant="outline" onClick={() => onOpenChange(false)}>Cancel</Button>
                    <Button onClick={handleSave}>Save Product</Button>
                </DialogFooter>
            </DialogContent>
        </Dialog>
    );
};


// --- Product Management Component ---
const ProductManager: React.FC<{ categories: Category[], onProductsUpdate: () => void }> = ({ categories, onProductsUpdate }) => {
    const [products, setProducts] = useState<Product[]>([]);
    const [loading, setLoading] = useState(true);
    const [isDialogOpen, setIsDialogOpen] = useState(false);
    const [editingProduct, setEditingProduct] = useState<Partial<Product> | null>(null);
    const [productToDelete, setProductToDelete] = useState<Product | null>(null);
    const [searchTerm, setSearchTerm] = useState('');

    const fetchProducts = useCallback(async () => {
        setLoading(true);
        const fetchedProducts = await getProducts();
        setProducts(fetchedProducts.sort((a, b) => a.name.localeCompare(b.name)));
        setLoading(false);
    }, []);

    useEffect(() => {
        fetchProducts();
    }, [fetchProducts]);
    
    const filteredProducts = useMemo(() => {
        if (!searchTerm.trim()) {
            return products;
        }
        const lowercasedTerm = searchTerm.trim().toLowerCase();
        return products.filter(p =>
            p.name.toLowerCase().includes(lowercasedTerm) ||
            p.category.toLowerCase().includes(lowercasedTerm)
        );
    }, [products, searchTerm]);

    const handleOpenDialog = (product: Partial<Product> | null = null) => {
        setEditingProduct(product ? { ...product } : { name: '', price: 0, description: '', image: '', category: categories[0]?.name || '', stock: 0, featured: false });
        setIsDialogOpen(true);
    };

    const handleSaveProduct = async (productData: Product) => {
        try {
            if (productData.id) {
                await updateProduct(productData);
            } else {
                const { id, ...newProductData } = productData;
                await addProduct(newProductData);
            }
            await fetchProducts();
            onProductsUpdate();
            setIsDialogOpen(false);
            setEditingProduct(null);
        } catch (error) {
            alert(error instanceof Error ? error.message : "An unknown error occurred.");
        }
    };
    
    const handleDeleteClick = (product: Product) => {
        setProductToDelete(product);
    };

    const handleConfirmDelete = async () => {
        if (productToDelete) {
            await deleteProduct(productToDelete.id);
            await fetchProducts();
            onProductsUpdate();
            setProductToDelete(null);
        }
    };

    if (loading) return <p>Loading products...</p>;

    return (
        <div>
            <div className="flex justify-between items-center mb-4">
                <h3 className="text-xl font-bold">Manage Products</h3>
                <Button onClick={() => handleOpenDialog()}><PlusCircle className="mr-2 h-4 w-4" /> Add Product</Button>
            </div>

            <div className="flex justify-between items-center mb-4 gap-4">
                <div className="flex-grow">
                    <Input 
                        placeholder="Search by product name or category..."
                        value={searchTerm}
                        onChange={(e) => setSearchTerm(e.target.value)}
                    />
                </div>
                <p className="text-sm text-gray-600 flex-shrink-0">
                    Showing {filteredProducts.length} of {products.length} products
                </p>
            </div>
            
            <ProductEditDialog open={isDialogOpen} onOpenChange={setIsDialogOpen} product={editingProduct} onSave={handleSaveProduct} categories={categories} />
            
            <Dialog open={!!productToDelete} onOpenChange={() => setProductToDelete(null)}>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>Confirm Deletion</DialogTitle>
                        <DialogDescription>
                            Are you sure you want to delete "{productToDelete?.name}"? This action cannot be undone.
                        </DialogDescription>
                    </DialogHeader>
                    <DialogFooter>
                        <Button variant="outline" onClick={() => setProductToDelete(null)}>Cancel</Button>
                        <Button variant="destructive" onClick={handleConfirmDelete}>Delete</Button>
                    </DialogFooter>
                </DialogContent>
            </Dialog>

            <div className="overflow-x-auto bg-white rounded-lg shadow">
                <table className="w-full text-sm text-left text-gray-500">
                    <thead className="text-xs text-gray-700 uppercase bg-gray-50">
                        <tr>
                            <th scope="col" className="px-6 py-3">Image</th>
                            <th scope="col" className="px-6 py-3">Name</th>
                            <th scope="col" className="px-6 py-3">Category</th>
                            <th scope="col" className="px-6 py-3">Price</th>
                            <th scope="col" className="px-6 py-3">Stock</th>
                            <th scope="col" className="px-6 py-3">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        {filteredProducts.length > 0 ? (
                            filteredProducts.map(product => (
                                <tr key={product.id} className="bg-white border-b">
                                    <td className="px-6 py-4"><img src={product.image} alt={product.name} className="h-12 w-12 object-cover rounded"/></td>
                                    <td className="px-6 py-4 font-medium text-gray-900">{product.name}</td>
                                    <td className="px-6 py-4">{product.category}</td>
                                    <td className="px-6 py-4">${product.price.toFixed(2)}</td>
                                    <td className="px-6 py-4">{product.stock}</td>
                                    <td className="px-6 py-4 flex space-x-2">
                                        <Button size="icon" variant="ghost" onClick={() => handleOpenDialog(product)}><Edit size={16} /></Button>
                                        <Button size="icon" variant="ghost" className="text-red-500 hover:text-red-700" onClick={() => handleDeleteClick(product)}><Trash2 size={16} /></Button>
                                    </td>
                                </tr>
                            ))
                        ) : (
                            <tr>
                                <td colSpan={6} className="text-center p-6 text-gray-500">
                                    {searchTerm ? `No products found matching "${searchTerm}".` : "No products available."}
                                </td>
                            </tr>
                        )}
                    </tbody>
                </table>
            </div>
        </div>
    );
};


// --- Order Management Component ---
const OrderManager: React.FC = () => {
    const [orders, setOrders] = useState<Order[]>([]);
    const [loading, setLoading] = useState(true);
    const [expandedOrderId, setExpandedOrderId] = useState<string | null>(null);

    const fetchOrders = useCallback(async () => {
        setLoading(true);
        const fetchedOrders = await getOrders();
        setOrders(fetchedOrders);
        setLoading(false);
    }, []);

    useEffect(() => {
        fetchOrders();
    }, [fetchOrders]);

    const handleStatusChange = async (orderId: string, status: OrderStatus) => {
        await updateOrderStatus(orderId, status);
        await fetchOrders();
    };
    
    const getStatusVariant = (status: OrderStatus) => {
        switch(status) {
            case 'Completed': return 'success';
            case 'Pending': return 'warning';
            case 'Cancelled': return 'destructive';
            default: return 'secondary';
        }
    }

    if (loading) return <p>Loading orders...</p>;
    if (orders.length === 0) return <p>No orders yet.</p>

    return (
        <div className="space-y-4">
            {orders.map(order => (
                <div key={order.id} className="bg-white rounded-lg shadow">
                    <button onClick={() => setExpandedOrderId(expandedOrderId === order.id ? null : order.id)} className="w-full text-left p-4">
                        <div className="flex justify-between items-start flex-wrap gap-4">
                            <div>
                                <h3 className="font-bold">Order #{order.id}</h3>
                                <p className="text-sm text-gray-500">{new Date(order.date).toLocaleString()}</p>
                                <p className="text-sm text-gray-500">{order.customerDetails.name}</p>
                            </div>
                            <div className="flex items-center gap-4">
                                <Badge variant={getStatusVariant(order.status)}>{order.status}</Badge>
                                <span className="font-bold text-lg">${order.total.toFixed(2)}</span>
                                { expandedOrderId === order.id ? <ChevronUp/> : <ChevronDown/> }
                            </div>
                        </div>
                    </button>
                    {expandedOrderId === order.id && (
                        <div className="p-4 border-t">
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                                <div>
                                    <h4 className="font-semibold mb-2">Customer Details</h4>
                                    <p><strong>Name:</strong> {order.customerDetails.name}</p>
                                    <p><strong>Email:</strong> {order.customerDetails.email}</p>
                                    <p><strong>Phone:</strong> {order.customerDetails.phone}</p>
                                    <p><strong>Address:</strong> {order.customerDetails.address}</p>
                                </div>
                                <div>
                                    <h4 className="font-semibold mb-2">Items</h4>
                                    <ul className="list-disc list-inside space-y-1">
                                    {order.items.map(item => <li key={item.id}>{item.name} (x{item.quantity})</li>)}
                                    </ul>
                                </div>
                            </div>
                             <div className="mt-4 flex justify-between items-center">
                                <div className="flex items-center gap-2">
                                     <label htmlFor={`status-${order.id}`} className="text-sm font-medium">Update Status:</label>
                                     <Select 
                                        id={`status-${order.id}`}
                                        value={order.status}
                                        onChange={(e) => handleStatusChange(order.id, e.target.value as OrderStatus)}
                                        className="w-48"
                                    >
                                        <option value="Pending">Pending</option>
                                        <option value="Completed">Completed</option>
                                        <option value="Cancelled">Cancelled</option>
                                    </Select>
                                </div>
                                {order.status === 'Pending' && <Button onClick={() => handleStatusChange(order.id, 'Completed')}>Process Order</Button>}
                            </div>
                        </div>
                    )}
                </div>
            ))}
        </div>
    );
};

// --- Category Management Component ---
const CategoryManager: React.FC<{onCategoryUpdate: () => void}> = ({onCategoryUpdate}) => {
    const [categories, setCategories] = useState<Category[]>([]);
    const [newCategoryName, setNewCategoryName] = useState('');
    const [newCategoryImage, setNewCategoryImage] = useState<string | null>(null);

    const fetchCats = useCallback(async () => {
        const fetched = await getCategories();
        setCategories(fetched);
    }, []);

    useEffect(() => {
        fetchCats();
    }, [fetchCats]);
    
    const handleImageSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0];
        if (file) {
            const reader = new FileReader();
            reader.onloadend = () => {
                setNewCategoryImage(reader.result as string);
            };
            reader.readAsDataURL(file);
        }
    };

    const handleAddCategory = async (e: React.FormEvent<HTMLFormElement>) => {
        e.preventDefault();
        if (!newCategoryName.trim()) return;
        await addCategory({ name: newCategoryName, image: newCategoryImage || undefined });
        setNewCategoryName('');
        setNewCategoryImage(null);
        // FIX: Assert e.currentTarget as HTMLFormElement to access elements.
        (e.currentTarget as HTMLFormElement).reset();
        await fetchCats();
        onCategoryUpdate();
    };

    const handleDeleteCategory = async (catId: string) => {
        if (window.confirm('Are you sure? This will not delete products in this category.')) {
            await deleteCategory(catId);
            await fetchCats();
            onCategoryUpdate();
        }
    }

    return (
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="bg-white p-6 rounded-lg shadow">
                <h3 className="font-bold text-lg mb-4">Add/Update Category</h3>
                <form onSubmit={handleAddCategory} className="space-y-4">
                    <Input value={newCategoryName} onChange={e => setNewCategoryName(e.target.value)} placeholder="Category Name"/>
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">Category Image (Optional)</label>
                        {newCategoryImage && <img src={newCategoryImage} alt="Preview" className="w-16 h-16 object-cover rounded mb-2"/>}
                        <Input type="file" accept="image/*" onChange={handleImageSelect} />
                    </div>
                    <Button type="submit" className="w-full">Save Category</Button>
                </form>
            </div>
             <div className="bg-white p-6 rounded-lg shadow">
                <h3 className="font-bold text-lg mb-4">Existing Categories ({categories.length})</h3>
                <ul className="space-y-2 max-h-60 overflow-y-auto pr-2">
                    {categories.map(cat => (
                        <li key={cat.id} className="flex justify-between items-center p-2 bg-gray-50 rounded">
                           <div className="flex items-center gap-3">
                               {cat.image && <img src={cat.image} alt={cat.name} className="w-10 h-10 object-cover rounded-md"/>}
                               <span>{cat.name} <span className="text-xs text-gray-500">({cat.productCount} products)</span></span>
                           </div>
                           <Button size="icon" variant="ghost" className="text-red-500 hover:text-red-700" onClick={() => handleDeleteCategory(cat.id)}>
                               <Trash2 size={16}/>
                           </Button>
                        </li>
                    ))}
                </ul>
            </div>
        </div>
    )
}

// --- Bulk Uploader Component ---
interface StagedImage {
    name: string;
    dataUrl: string;
}
const BulkUploader: React.FC<{ categories: Category[], onUploadComplete: () => void }> = ({ categories, onUploadComplete }) => {
    const [stagedImages, setStagedImages] = useState<StagedImage[]>([]);
    const [editingProduct, setEditingProduct] = useState<Partial<Product> | null>(null);
    const [isDialogOpen, setIsDialogOpen] = useState(false);

    const handleFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
        const files = Array.from(e.target.files || []);
        // FIX: Explicitly type `file` as `File` to prevent it from being inferred as `unknown`, causing a type error.
        files.forEach((file: File) => {
            const reader = new FileReader();
            reader.onload = (event) => {
                setStagedImages(prev => [...prev, { name: file.name, dataUrl: event.target?.result as string }]);
            };
            reader.readAsDataURL(file);
        });
    };

    const handleOpenDialog = (image: StagedImage) => {
        setEditingProduct({
            name: image.name.split('.').slice(0, -1).join('.'), // filename without extension
            image: image.dataUrl,
            price: 0,
            stock: 0,
            description: '',
            category: categories[0]?.name || '',
            featured: false,
        });
        setIsDialogOpen(true);
    };

    const handleSaveProduct = async (productData: Product) => {
        try {
            const { id, ...newProductData } = productData;
            await addProduct(newProductData);
            setStagedImages(prev => prev.filter(img => img.dataUrl !== productData.image));
            setIsDialogOpen(false);
            setEditingProduct(null);
            onUploadComplete();
        } catch(error) {
            alert(error instanceof Error ? error.message : "An unknown error occurred.");
        }
    };

    return (
        <div className="bg-white p-6 rounded-lg shadow">
            <h3 className="font-bold text-lg mb-4">Upload Product Images</h3>
            <div className="border-2 border-dashed border-gray-300 rounded-lg p-8 text-center">
                 <UploadCloud className="mx-auto h-12 w-12 text-gray-400" />
                 <label htmlFor="file-upload" className="relative cursor-pointer rounded-md font-semibold text-brand-purple hover:text-brand-purple/80">
                    <span>Upload files</span>
                    <input id="file-upload" name="file-upload" type="file" multiple accept="image/*" className="sr-only" onChange={handleFileSelect} />
                </label>
                <p className="text-xs leading-5 text-gray-600">PNG, JPG, GIF up to 10MB</p>
            </div>
            {stagedImages.length > 0 && (
                <div className="mt-6">
                    <h4 className="font-semibold mb-2">Staged Images ({stagedImages.length})</h4>
                    <div className="grid grid-cols-2 sm:grid-cols-4 md:grid-cols-6 lg:grid-cols-8 gap-4">
                        {stagedImages.map(image => (
                            <button key={image.dataUrl} onClick={() => handleOpenDialog(image)} className="relative aspect-square group">
                                <img src={image.dataUrl} alt={image.name} className="w-full h-full object-cover rounded-lg"/>
                                <div className="absolute inset-0 bg-black/50 flex items-center justify-center text-white opacity-0 group-hover:opacity-100 transition-opacity rounded-lg">
                                    <Edit size={24}/>
                                </div>
                            </button>
                        ))}
                    </div>
                </div>
            )}
             <ProductEditDialog open={isDialogOpen} onOpenChange={setIsDialogOpen} product={editingProduct} onSave={handleSaveProduct} categories={categories} />
        </div>
    )
}

// --- Site Settings Component ---
const SiteManager = () => {
    const { logo, sliderImages, updateLogo, updateSliderImages, resetBranding } = useBranding();

    const handleLogoUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0];
        if (file) {
            const reader = new FileReader();
            reader.onloadend = () => {
                updateLogo(reader.result as string);
            };
            reader.readAsDataURL(file);
        }
    };
    
    const handleSliderUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
        const files = Array.from(e.target.files || []);
        const newImages: string[] = [];
        let filesProcessed = 0;

        if (files.length === 0) return;

        files.forEach(file => {
            const reader = new FileReader();
            reader.onloadend = () => {
                newImages.push(reader.result as string);
                filesProcessed++;
                if (filesProcessed === files.length) {
                    updateSliderImages([...sliderImages, ...newImages]);
                }
            };
            reader.readAsDataURL(file);
        });
    };
    
    const handleRemoveSliderImage = (indexToRemove: number) => {
        const newSliderImages = sliderImages.filter((_, index) => index !== indexToRemove);
        updateSliderImages(newSliderImages);
    };
    
    const handleReset = () => {
        if(window.confirm('Are you sure you want to reset the logo and slider images to their default settings?')) {
            resetBranding();
        }
    }

    return (
        <div className="space-y-8 bg-white p-6 rounded-lg shadow">
            <div>
                <h3 className="text-xl font-bold mb-4 border-b pb-2">Site Logo & Favicon</h3>
                <div className="flex items-center gap-6">
                    <div>
                        <p className="text-sm font-medium mb-2">Current Logo:</p>
                        <img src={logo} alt="Current Logo" className="h-24 w-auto bg-gray-100 p-2 rounded-lg border" />
                    </div>
                    <div>
                        <label htmlFor="logo-upload" className="block text-sm font-medium text-gray-700 mb-2">Upload new logo:</label>
                        <Input id="logo-upload" type="file" accept="image/*" onChange={handleLogoUpload} />
                    </div>
                </div>
            </div>
            <div>
                <h3 className="text-xl font-bold mb-4 border-b pb-2">Homepage Slider Images</h3>
                <div className="mb-4">
                    <label htmlFor="slider-upload" className="block text-sm font-medium text-gray-700 mb-2">Add images to slider:</label>
                    <Input id="slider-upload" type="file" multiple accept="image/*" onChange={handleSliderUpload} />
                </div>
                {sliderImages.length > 0 ? (
                    <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 gap-4">
                        {sliderImages.map((image, index) => (
                            <div key={index} className="relative group">
                                <img src={image} alt={`Slider image ${index + 1}`} className="w-full h-32 object-cover rounded-lg border"/>
                                <Button size="icon" variant="destructive" className="absolute top-1 right-1 h-7 w-7 opacity-0 group-hover:opacity-100 transition-opacity" onClick={() => handleRemoveSliderImage(index)}>
                                    <X size={16}/>
                                </Button>
                            </div>
                        ))}
                    </div>
                ) : (
                    <p className="text-sm text-gray-500">No custom slider images uploaded. The slider will use category images as a fallback.</p>
                )}
            </div>
            <div>
                <h3 className="text-xl font-bold mb-4 border-b pb-2">Danger Zone</h3>
                <Button variant="destructive" onClick={handleReset}>
                    <RotateCcw className="mr-2 h-4 w-4" /> Reset Branding
                </Button>
                 <p className="text-sm text-gray-500 mt-2">This will remove the custom logo and all slider images, reverting to the site defaults.</p>
            </div>
        </div>
    );
};


// --- Main Admin Page Component ---
const AdminPage: React.FC = () => {
    useDocumentTitle('Admin Dashboard - Lanisdeb Market');
    const { logout } = useAuth();
    const [categories, setCategories] = useState<Category[]>([]);

    const fetchAllCategories = useCallback(async () => {
        const cats = await getCategories();
        setCategories(cats);
    }, []);

    useEffect(() => {
        fetchAllCategories();
    }, [fetchAllCategories]);
    
    return (
        <div className="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
            <header className="flex justify-between items-center mb-8">
                <h1 className="text-3xl font-bold font-heading">Admin Dashboard</h1>
                <Button variant="outline" onClick={logout}><LogOut className="mr-2 h-4 w-4" /> Logout</Button>
            </header>
            
            <Tabs defaultValue="products">
                <TabsList className="grid w-full grid-cols-2 md:grid-cols-5">
                    <TabsTrigger value="products">Products</TabsTrigger>
                    <TabsTrigger value="orders">Orders</TabsTrigger>
                    <TabsTrigger value="categories">Categories</TabsTrigger>
                    <TabsTrigger value="bulk-upload">Bulk Upload</TabsTrigger>
                    <TabsTrigger value="settings">Site Settings</TabsTrigger>
                </TabsList>
                <div className="bg-brand-bg-light rounded-b-lg shadow p-6 mt-2">
                    <TabsContent value="products">
                       <ProductManager categories={categories} onProductsUpdate={fetchAllCategories}/>
                    </TabsContent>
                    <TabsContent value="orders">
                        <OrderManager />
                    </TabsContent>
                    <TabsContent value="categories">
                        <CategoryManager onCategoryUpdate={fetchAllCategories}/>
                    </TabsContent>
                    <TabsContent value="bulk-upload">
                        <BulkUploader categories={categories} onUploadComplete={() => { fetchAllCategories(); }}/>
                    </TabsContent>
                    <TabsContent value="settings">
                        <SiteManager />
                    </TabsContent>
                </div>
            </Tabs>
        </div>
    );
};

export default AdminPage;