
import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { useCart } from '../context/CartContext';
import { Minus, Plus, Trash2 } from 'lucide-react';
import { Button, Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter } from '../components/ui';
import { CartItem } from '../types';

const CartPage: React.FC = () => {
  const { cartItems, cartTotal, updateQuantity, removeFromCart } = useCart();
  const [itemToRemove, setItemToRemove] = useState<CartItem | null>(null);
  const [highlightedId, setHighlightedId] = useState<string | null>(null);

  const handleUpdateQuantity = (productId: string, newQuantity: number) => {
    updateQuantity(productId, newQuantity);
    setHighlightedId(productId);
    setTimeout(() => setHighlightedId(null), 600);
  };
  
  const handleConfirmRemove = () => {
    if (itemToRemove) {
      removeFromCart(itemToRemove.id);
    }
    setItemToRemove(null);
  };

  if (cartItems.length === 0) {
    return (
      <div className="container mx-auto text-center py-20 px-4">
        <h1 className="text-4xl font-bold font-heading text-brand-text-dark">Your Cart is Empty</h1>
        <p className="text-lg text-brand-text-light mt-4">Looks like you haven't added anything to your cart yet.</p>
        <Button asChild size="lg" className="mt-8">
          <Link to="/shop">Start Shopping</Link>
        </Button>
      </div>
    );
  }

  return (
    <>
      <Dialog open={!!itemToRemove} onOpenChange={(isOpen) => !isOpen && setItemToRemove(null)}>
        <DialogContent>
            <DialogHeader>
                <DialogTitle>Remove Item</DialogTitle>
                <DialogDescription>
                    Are you sure you want to remove "{itemToRemove?.name}" from your cart?
                </DialogDescription>
            </DialogHeader>
            <DialogFooter>
                <Button variant="outline" onClick={() => setItemToRemove(null)}>Cancel</Button>
                <Button variant="destructive" onClick={handleConfirmRemove}>Remove</Button>
            </DialogFooter>
        </DialogContent>
      </Dialog>
      <div className="bg-brand-bg-light min-h-[calc(100vh-10rem)]">
        <div className="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
          <h1 className="text-4xl font-bold font-heading text-brand-text-dark mb-8 text-center">Shopping Cart</h1>
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
            <div className="lg:col-span-2 bg-white rounded-lg shadow p-6 space-y-4">
              {cartItems.map(item => (
                <div key={item.id} className="flex items-center space-x-4 border-b pb-4 last:border-b-0">
                  <img src={item.image} alt={item.name} className="w-24 h-24 object-cover rounded-lg" />
                  <div className="flex-grow">
                    <Link to={`/products/${item.id}`} className="font-bold text-brand-text-dark hover:text-brand-purple">{item.name}</Link>
                    <p className="text-sm text-brand-text-light">${item.price.toFixed(2)}</p>
                  </div>
                  <div className="flex items-center border border-gray-300 rounded-lg">
                    <Button variant="outline" size="icon" className="rounded-r-none h-9 w-9" onClick={() => handleUpdateQuantity(item.id, item.quantity - 1)}><Minus size={14} /></Button>
                    <span className={`px-4 font-bold ${highlightedId === item.id ? 'animate-scale-up' : ''}`}>{item.quantity}</span>
                    <Button variant="outline" size="icon" className="rounded-l-none h-9 w-9" onClick={() => handleUpdateQuantity(item.id, item.quantity + 1)} disabled={item.quantity >= item.stock}><Plus size={14} /></Button>
                  </div>
                  <p className={`font-bold w-24 text-right rounded px-1 ${highlightedId === item.id ? 'animate-flash-bg' : ''}`}>${(item.price * item.quantity).toFixed(2)}</p>
                  <Button variant="ghost" size="icon" onClick={() => setItemToRemove(item)} className="text-gray-500 hover:text-red-500"><Trash2 size={20} /></Button>
                </div>
              ))}
            </div>
            <div className="lg:col-span-1">
              <div className="bg-white rounded-lg shadow p-6 sticky top-24">
                <h2 className="text-2xl font-bold font-heading border-b pb-4 mb-4">Order Summary</h2>
                <div className="space-y-2 mb-4">
                  <div className="flex justify-between">
                    <span>Subtotal</span>
                    <span>${cartTotal.toFixed(2)}</span>
                  </div>
                  <div className="flex justify-between">
                    <span>Shipping</span>
                    <span>Calculated at checkout</span>
                  </div>
                </div>
                <div className="flex justify-between font-bold text-lg border-t pt-4">
                  <span>Total</span>
                  <span className={`px-1 rounded ${highlightedId ? 'animate-flash-bg' : ''}`}>${cartTotal.toFixed(2)}</span>
                </div>
                <Button asChild size="lg" className="w-full mt-6">
                  <Link to="/checkout">Proceed to Checkout</Link>
                </Button>
              </div>
            </div>
          </div>
        </div>
      </div>
    </>
  );
};

export default CartPage;