
import React, { useState } from 'react';
import { useCart } from '../context/CartContext';
import { useNavigate, Navigate } from 'react-router-dom';
import { CustomerDetails, Order } from '../types';
import { addOrder } from '../data/db';
import { Button, Input, Textarea } from '../components/ui';
import { CreditCard, Landmark } from 'lucide-react';

const PaymentOption = ({ value, selected, onSelect, title, icon: Icon, children }: React.PropsWithChildren<{ value: string, selected: boolean, onSelect: (value: string) => void, title: string, icon: React.ElementType }>) => (
    <div 
        onClick={() => onSelect(value)}
        className={`border p-4 rounded-lg cursor-pointer transition-all flex items-center ${selected ? 'border-brand-purple ring-2 ring-brand-purple' : 'border-gray-300 hover:border-gray-400'}`}
      >
        <input 
          id={value} 
          name="payment" 
          type="radio" 
          checked={selected} 
          readOnly
          className="focus:ring-brand-purple h-4 w-4 text-brand-purple border-gray-300" 
        />
        <label htmlFor={value} className="ml-3 flex items-center gap-3 text-sm font-medium text-gray-700 cursor-pointer">
            <Icon className="h-6 w-6 text-gray-600" />
            {title}
        </label>
    </div>
);

const StripeForm = () => (
    <div className="space-y-3 p-4 border rounded-lg bg-gray-50 animate-fade-in">
        <h4 className="font-semibold text-sm text-gray-800">Enter Card Details</h4>
        <div>
            <Input type="text" name="cardNumber" id="cardNumber" placeholder="Card Number" autoComplete="cc-number" />
        </div>
        <div className="grid grid-cols-2 gap-3">
            <div>
                <Input type="text" name="expiry" id="expiry" placeholder="MM / YY" autoComplete="cc-exp" />
            </div>
            <div>
                <Input type="text" name="cvc" id="cvc" placeholder="CVC" autoComplete="cc-csc" />
            </div>
        </div>
         <style>{`
            @keyframes fade-in {
              0% { opacity: 0; transform: translateY(-10px); }
              100% { opacity: 1; transform: translateY(0); }
            }
            .animate-fade-in {
              animation: fade-in 0.3s ease-out;
            }
        `}</style>
    </div>
);


const CheckoutPage: React.FC = () => {
  const { cartItems, cartTotal, clearCart, showToast } = useCart();
  const navigate = useNavigate();
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [customerDetails, setCustomerDetails] = useState<CustomerDetails>({
    name: '', email: '', phone: '', address: ''
  });
  const [paymentMethod, setPaymentMethod] = useState('pod');

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setCustomerDetails(prev => ({ ...prev, [name]: value }));
  };

  const paymentMethods: { [key: string]: { name: string, icon: React.ElementType, isExternal: boolean } } = {
      pod: { name: 'Pay on Delivery', icon: Landmark, isExternal: false },
      stripe: { name: 'Credit/Debit Card', icon: CreditCard, isExternal: true },
      paypal: { name: 'PayPal', icon: Landmark, isExternal: true },
      paystack: { name: 'Paystack', icon: Landmark, isExternal: true },
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsSubmitting(true);
    
    const selectedPayment = paymentMethods[paymentMethod];

    // Simulate payment processing for external providers
    if (selectedPayment.isExternal) {
        showToast(`Processing payment via ${selectedPayment.name}...`);
        await new Promise(resolve => setTimeout(resolve, 2500)); // Simulate API call to processor
    }
    
    // For this demo, we'll assume the payment is always successful
    const paymentSuccessful = true;
    const paymentId = selectedPayment.isExternal ? `txn_${new Date().getTime()}` : undefined;

    if (paymentSuccessful) {
        const orderDetails: Omit<Order, 'id' | 'date'> = {
          items: cartItems,
          total: cartTotal,
          customerDetails,
          paymentMethod: selectedPayment.name,
          paymentId,
          status: 'Pending',
        };

        try {
          const newOrder = await addOrder(orderDetails);
          clearCart();
          showToast(`Admin notified of new order #${newOrder.id}`);
          navigate('/order-success', { state: { order: newOrder } });
        } catch (error) {
          console.error("Failed to place order:", error);
          showToast("There was an error placing your order. Please try again.");
          setIsSubmitting(false);
        }
    } else {
        showToast("Payment failed. Please check your details and try again.");
        setIsSubmitting(false);
    }
  };

  if (cartItems.length === 0) {
    return <Navigate to="/shop" replace />;
  }

  return (
    <div className="bg-brand-bg-light">
      <div className="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
        <h1 className="text-4xl font-bold font-heading text-brand-text-dark mb-8 text-center">Checkout</h1>
        <form onSubmit={handleSubmit} className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          <div className="lg:col-span-2 bg-white rounded-lg shadow p-6">
            <h2 className="text-2xl font-bold font-heading mb-6">Shipping Information</h2>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <label htmlFor="name" className="block text-sm font-medium text-gray-700 mb-1">Full Name</label>
                <Input type="text" name="name" id="name" required value={customerDetails.name} onChange={handleInputChange} />
              </div>
              <div>
                <label htmlFor="email" className="block text-sm font-medium text-gray-700 mb-1">Email Address</label>
                <Input type="email" name="email" id="email" required value={customerDetails.email} onChange={handleInputChange} />
              </div>
              <div className="md:col-span-2">
                <label htmlFor="phone" className="block text-sm font-medium text-gray-700 mb-1">Phone Number</label>
                <Input type="tel" name="phone" id="phone" required value={customerDetails.phone} onChange={handleInputChange} />
              </div>
              <div className="md:col-span-2">
                <label htmlFor="address" className="block text-sm font-medium text-gray-700 mb-1">Full Address</label>
                <Textarea name="address" id="address" rows={3} required value={customerDetails.address} onChange={handleInputChange} />
              </div>
            </div>
          </div>
          <div className="lg:col-span-1">
            <div className="bg-white rounded-lg shadow p-6 sticky top-24">
              <h2 className="text-2xl font-bold font-heading border-b pb-4 mb-4">Your Order</h2>
              <div className="space-y-2 mb-4 max-h-60 overflow-y-auto pr-2">
                {cartItems.map(item => (
                  <div key={item.id} className="flex justify-between items-start text-sm">
                    <p className="font-semibold">{item.name} <span className="text-gray-500">x{item.quantity}</span></p>
                    <p>${(item.price * item.quantity).toFixed(2)}</p>
                  </div>
                ))}
              </div>
              <div className="flex justify-between font-bold text-lg border-t pt-4 mb-6">
                <span>Total</span>
                <span>${cartTotal.toFixed(2)}</span>
              </div>
              <h3 className="text-xl font-bold font-heading mb-4">Payment Method</h3>
              <div className="space-y-3">
                {Object.entries(paymentMethods).map(([key, { name, icon }]) => (
                    <PaymentOption key={key} value={key} selected={paymentMethod === key} onSelect={setPaymentMethod} title={name} icon={icon} />
                ))}
              </div>
              
              <div className="mt-4 min-h-[100px]">
                  {paymentMethod === 'stripe' && <StripeForm />}
                  {(paymentMethod === 'paypal' || paymentMethod === 'paystack') && (
                     <div className="p-4 border rounded-lg bg-gray-50 animate-fade-in">
                        <p className="text-sm text-center text-gray-600">You will be redirected to {paymentMethods[paymentMethod].name} to complete your payment.</p>
                     </div>
                  )}
              </div>

              <Button type="submit" size="lg" className="w-full mt-6" disabled={isSubmitting}>
                {isSubmitting ? 'Processing...' : `Place Order`}
              </Button>
            </div>
          </div>
        </form>
      </div>
    </div>
  );
};

export default CheckoutPage;