import React, { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { Product, Category } from '../types';
import { getFeaturedProducts, getCategories } from '../data/db';
import { useBranding } from '../context/BrandingContext';
import ProductCard, { ProductCardSkeleton } from '../components/ProductCard';
import { Button } from '../components/ui';
import useDocumentTitle from '../hooks/useDocumentTitle';

const HeroSlider: React.FC = () => {
    const { sliderImages } = useBranding();
    const [categories, setCategories] = useState<Category[]>([]);
    const [loadingCategories, setLoadingCategories] = useState(true);
    const [currentIndex, setCurrentIndex] = useState(0);
    
    const hasCustomImages = sliderImages.length > 0;
    const slides = hasCustomImages ? sliderImages.map(url => ({ id: url, name: "Featured", image: url })) : categories.filter(c => c.image);

    useEffect(() => {
        if (!hasCustomImages) {
            getCategories().then(cats => {
                setCategories(cats);
                setLoadingCategories(false);
            });
        } else {
            setLoadingCategories(false);
        }
    }, [hasCustomImages]);

    useEffect(() => {
        if (slides.length === 0) return;
        const interval = setInterval(() => {
            setCurrentIndex(prevIndex => (prevIndex + 1) % slides.length);
        }, 5000);
        return () => clearInterval(interval);
    }, [slides.length]);
    
    if (loadingCategories) {
      return <div className="h-[60vh] md:h-[80vh] w-full bg-brand-bg-light animate-pulse" />;
    }

    if (slides.length === 0) return (
      <section className="bg-gradient-to-r from-brand-gold/20 via-brand-purple/5 to-white py-20 md:py-32">
        <div className="container mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <h1 className="text-4xl md:text-6xl font-extrabold font-heading text-brand-text-dark tracking-tight">
            Authentic African & Caribbean Flavors
          </h1>
          <p className="mt-4 max-w-2xl mx-auto text-lg md:text-xl text-brand-text-light">
            Discover a wide range of groceries, spices, and ingredients delivered right to your door.
          </p>
          <div className="mt-8">
            <Button asChild size="lg" variant="hero">
              <Link to="/shop">Shop Now</Link>
            </Button>
          </div>
        </div>
      </section>
    );

    return (
        <section className="relative w-full h-[60vh] md:h-[80vh] overflow-hidden bg-brand-bg-light">
            {slides.map((slide, index) => (
                <div
                    key={slide.id}
                    className="absolute inset-0 w-full h-full transition-opacity duration-1000 ease-in-out"
                    style={{ opacity: index === currentIndex ? 1 : 0 }}
                >
                    <img src={slide.image} alt={slide.name} className="w-full h-full object-cover" />
                    <div className="absolute inset-0 bg-black/50 flex items-center justify-center">
                        <div className="text-center text-white p-4">
                            <h1 className="text-4xl md:text-6xl font-extrabold font-heading tracking-tight">
                                {hasCustomImages ? "Authentic Flavors, Delivered" : `Explore ${slide.name}`}
                            </h1>
                            <p className="mt-4 max-w-2xl mx-auto text-lg md:text-xl">
                                {hasCustomImages ? "Discover our collection of premium groceries." : `Discover our collection of authentic ${slide.name.toLowerCase()}.`}
                            </p>
                            <div className="mt-8">
                                <Button asChild size="lg" variant="hero">
                                    <Link to={hasCustomImages ? "/shop" : `/shop?category=${encodeURIComponent(slide.name)}`}>Shop Now</Link>
                                </Button>
                            </div>
                        </div>
                    </div>
                </div>
            ))}
             <div className="absolute bottom-5 left-1/2 -translate-x-1/2 flex space-x-2 z-10">
                {slides.map((_, index) => (
                    <button
                        key={index}
                        onClick={() => setCurrentIndex(index)}
                        className={`w-3 h-3 rounded-full transition-colors ${currentIndex === index ? 'bg-white' : 'bg-white/50 hover:bg-white/75'}`}
                        aria-label={`Go to slide ${index + 1}`}
                    />
                ))}
            </div>
        </section>
    );
};


const HomePage: React.FC = () => {
  useDocumentTitle('Lanisdeb Market - Authentic African & Caribbean Groceries');
  const [featuredProducts, setFeaturedProducts] = useState<Product[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        const products = await getFeaturedProducts();
        setFeaturedProducts(products);
      } catch (error) {
        console.error("Failed to fetch page data:", error);
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, []);

  return (
    <div>
      <HeroSlider />
      <section className="py-16 bg-brand-bg-light">
        <div className="container mx-auto px-4 sm:px-6 lg:px-8">
          <h2 className="text-3xl font-bold font-heading text-center text-brand-text-dark mb-8">Featured Products</h2>
          {loading ? (
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-8">
              {Array.from({ length: 4 }).map((_, index) => <ProductCardSkeleton key={index} />)}
            </div>
          ) : (
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-8">
              {featuredProducts.map(product => (
                <ProductCard key={product.id} product={product} />
              ))}
            </div>
          )}
           <div className="text-center mt-12">
            <Button asChild size="lg" variant="outline">
              <Link to="/shop">View All Products</Link>
            </Button>
          </div>
        </div>
      </section>
      
      <section className="py-16">
        <div className="container mx-auto px-4 sm:px-6 lg:px-8">
          <h2 className="text-3xl font-bold font-heading text-center text-brand-text-dark mb-12">Why Shop With Us?</h2>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8 text-center">
            <div className="p-6">
              <h3 className="text-xl font-bold font-heading text-brand-purple mb-2">Quality & Authenticity</h3>
              <p className="text-brand-text-light">We source the best products to bring you the true taste of Africa and the Caribbean.</p>
            </div>
            <div className="p-6">
              <h3 className="text-xl font-bold font-heading text-brand-purple mb-2">Convenient Shopping</h3>
              <p className="text-brand-text-light">Easy online ordering and fast delivery to your doorstep.</p>
            </div>
            <div className="p-6">
              <h3 className="text-xl font-bold font-heading text-brand-purple mb-2">Community Focused</h3>
              <p className="text-brand-text-light">A family-owned business dedicated to serving our community's needs.</p>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
};

export default HomePage;