
import React, { useState, useEffect } from 'react';
import { useParams, Link, useNavigate } from 'react-router-dom';
import { Product } from '../types';
import { getProductById, getProducts } from '../data/db';
import { useCart } from '../context/CartContext';
import { Button, Skeleton } from '../components/ui';
import { Minus, Plus, ChevronLeft } from 'lucide-react';
import ProductCard, { ProductCardSkeleton } from '../components/ProductCard';
import useDocumentTitle from '../hooks/useDocumentTitle';

const ProductDetailPageSkeleton: React.FC = () => (
    <div className="bg-brand-bg-light">
        <div className="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
            <div className="mb-6">
                <Skeleton className="h-6 w-32" />
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-8 bg-white p-8 rounded-lg shadow-md">
                <div>
                    <Skeleton className="w-full aspect-square max-h-[500px] rounded-lg" />
                </div>
                <div>
                    <Skeleton className="h-4 w-1/4 mb-4" />
                    <Skeleton className="h-10 w-3/4 mb-4" />
                    <Skeleton className="h-8 w-1/3 mb-6" />
                    <Skeleton className="h-5 w-full mb-2" />
                    <Skeleton className="h-5 w-full mb-2" />
                    <Skeleton className="h-5 w-2/3 mb-6" />
                    
                    <div className="flex items-center space-x-4 mb-6">
                        <Skeleton className="h-11 w-11 rounded-l-lg" />
                        <Skeleton className="h-6 w-12" />
                        <Skeleton className="h-11 w-11 rounded-r-lg" />
                    </div>
                    
                    <Skeleton className="h-5 w-1/4 mb-6" />
                    
                    <Skeleton className="h-12 w-48 rounded-md" />
                </div>
            </div>

            <div className="mt-16">
                <h2 className="text-3xl font-bold font-heading text-center text-brand-text-dark mb-8">You Might Also Like</h2>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-8">
                {Array.from({ length: 4 }).map((_, index) => <ProductCardSkeleton key={index} />)}
                </div>
            </div>
        </div>
    </div>
);


const ProductDetailPage: React.FC = () => {
  const { productId } = useParams<{ productId: string }>();
  const navigate = useNavigate();
  const { addToCart } = useCart();
  
  const [product, setProduct] = useState<Product | null>(null);
  const [relatedProducts, setRelatedProducts] = useState<Product[]>([]);
  const [quantity, setQuantity] = useState(1);
  const [loading, setLoading] = useState(true);

  useDocumentTitle(product ? product.name : 'Loading Product...');

  useEffect(() => {
    const fetchProduct = async () => {
      if (!productId) return;
      try {
        setLoading(true);
        const fetchedProduct = await getProductById(productId);
        if (fetchedProduct) {
          setProduct(fetchedProduct);
          const allProducts = await getProducts();
          const related = allProducts
            .filter(p => p.category === fetchedProduct.category && p.id !== fetchedProduct.id)
            .slice(0, 4);
          setRelatedProducts(related);
        } else {
          navigate('/404');
        }
      } catch (error) {
        console.error("Failed to fetch product:", error);
      } finally {
        setLoading(false);
      }
    };
    fetchProduct();
    setQuantity(1);
    window.scrollTo(0, 0);
  }, [productId, navigate]);

  const handleQuantityChange = (amount: number) => {
    if (!product) return;
    setQuantity(prev => {
      const newQuantity = prev + amount;
      if (newQuantity < 1) return 1;
      if (newQuantity > product.stock) return product.stock;
      return newQuantity;
    });
  };
  
  const handleAddToCart = () => {
    if (product) {
      addToCart(product, quantity);
    }
  };

  if (loading) {
    return <ProductDetailPageSkeleton />;
  }

  if (!product) {
    return <div className="text-center py-20">Product not found.</div>;
  }

  return (
    <div className="bg-brand-bg-light">
      <div className="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
        <div className="mb-6">
            <Link to="/shop" className="inline-flex items-center text-brand-purple hover:underline">
                <ChevronLeft size={20} className="mr-1" />
                Back to Shop
            </Link>
        </div>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-8 bg-white p-8 rounded-lg shadow-md">
          {/* Product Image */}
          <div>
            <img src={product.image} alt={product.name} className="w-full h-auto max-h-[500px] object-contain rounded-lg" />
          </div>
          {/* Product Details */}
          <div>
            <span className="text-sm font-semibold text-brand-purple">{product.category}</span>
            <h1 className="text-4xl font-bold font-heading text-brand-text-dark mt-2 mb-4">{product.name}</h1>
            <p className="text-3xl font-bold text-brand-purple mb-6">${product.price.toFixed(2)}</p>
            <p className="text-brand-text-light mb-6">{product.description}</p>
            
            <div className="flex items-center space-x-4 mb-6">
              <p>Quantity:</p>
              <div className="flex items-center border border-gray-300 rounded-lg">
                <Button variant="outline" size="icon" className="rounded-r-none h-11 w-11" onClick={() => handleQuantityChange(-1)}><Minus size={16} /></Button>
                <span className="px-6 font-bold text-lg">{quantity}</span>
                <Button variant="outline" size="icon" className="rounded-l-none h-11 w-11" onClick={() => handleQuantityChange(1)} disabled={quantity >= product.stock}><Plus size={16} /></Button>
              </div>
            </div>
            
            <p className="text-sm text-gray-600 mb-6">
              {product.stock > 0 ? `In Stock (${product.stock} available)` : 'Out of Stock'}
            </p>
            
            <Button size="lg" onClick={handleAddToCart} disabled={product.stock === 0} className="w-full md:w-auto">
              {product.stock > 0 ? 'Add to Cart' : 'Out of Stock'}
            </Button>
          </div>
        </div>

        {/* Related Products */}
        {relatedProducts.length > 0 && (
          <div className="mt-16">
            <h2 className="text-3xl font-bold font-heading text-center text-brand-text-dark mb-8">You Might Also Like</h2>
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-8">
              {relatedProducts.map(related => (
                <ProductCard key={related.id} product={related} />
              ))}
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default ProductDetailPage;