

import React, { useState, useEffect, useMemo } from 'react';
import { useLocation } from 'react-router-dom';
import { Product, Category } from '../types';
import { getProducts, getCategories } from '../data/db';
import ProductCard, { ProductCardSkeleton } from '../components/ProductCard';
import useDocumentTitle from '../hooks/useDocumentTitle';
import { Select, SelectValue, Input, Skeleton } from '../components/ui';

const FilterSidebarSkeleton: React.FC = () => (
    <div className="bg-white p-6 rounded-lg shadow sticky top-24">
        <Skeleton className="h-6 w-1/3 mb-4" />
        <div className="mb-6">
            <Skeleton className="h-4 w-1/4 mb-1" />
            <Skeleton className="h-10 w-full" />
        </div>
        <div>
            <Skeleton className="h-5 w-1/2 mb-2" />
            <div className="space-y-2">
                <Skeleton className="h-8 w-full" />
                <Skeleton className="h-8 w-full" />
                <Skeleton className="h-8 w-full" />
                <Skeleton className="h-8 w-full" />
            </div>
        </div>
    </div>
);

const ShopPage: React.FC = () => {
  useDocumentTitle('Shop - Lanisdeb Market');
  const location = useLocation();
  const [products, setProducts] = useState<Product[]>([]);
  const [categories, setCategories] = useState<Category[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedCategory, setSelectedCategory] = useState('All');
  const [searchTerm, setSearchTerm] = useState('');
  const [sortOrder, setSortOrder] = useState('name-asc');
  const [suggestions, setSuggestions] = useState<Product[]>([]);
  const [isSuggestionsVisible, setIsSuggestionsVisible] = useState(false);

  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        const [fetchedProducts, fetchedCategories] = await Promise.all([
          getProducts(),
          getCategories(),
        ]);
        setProducts(fetchedProducts);
        setCategories([{ id: 'all', name: 'All' }, ...fetchedCategories]);
        
        const params = new URLSearchParams(location.search);
        const categoryFromUrl = params.get('category');
        if (categoryFromUrl && fetchedCategories.some(c => c.name === categoryFromUrl)) {
            setSelectedCategory(categoryFromUrl);
        }

      } catch (error) {
        console.error("Failed to fetch shop data:", error);
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, [location.search]);

  const handleSearchChange = (e: React.ChangeEvent<HTMLInputElement>) => {
      const value = e.target.value;
      setSearchTerm(value);
      if (value.length > 1) {
          const filtered = products.filter(p =>
              p.name.toLowerCase().includes(value.toLowerCase()) ||
              p.description.toLowerCase().includes(value.toLowerCase())
          ).slice(0, 5);
          setSuggestions(filtered);
          setIsSuggestionsVisible(true);
      } else {
          setSuggestions([]);
          setIsSuggestionsVisible(false);
      }
  };

  const handleSuggestionClick = (product: Product) => {
      setSearchTerm(product.name);
      setSuggestions([]);
      setIsSuggestionsVisible(false);
  };

  const filteredAndSortedProducts = useMemo(() => {
    let filtered = products;

    if (selectedCategory !== 'All') {
      filtered = filtered.filter(p => p.category === selectedCategory);
    }

    if (searchTerm) {
      filtered = filtered.filter(p =>
        p.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        p.description.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    return filtered.sort((a, b) => {
      switch (sortOrder) {
        case 'price-asc': return a.price - b.price;
        case 'price-desc': return b.price - a.price;
        case 'name-desc': return b.name.localeCompare(a.name);
        case 'name-asc':
        default:
          return a.name.localeCompare(b.name);
      }
    });
  }, [products, selectedCategory, searchTerm, sortOrder]);

  return (
    <div className="bg-brand-bg-light min-h-screen">
      <div className="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
        <div className="text-center mb-12">
            <h1 className="text-4xl font-bold font-heading text-brand-text-dark">Our Products</h1>
            <p className="mt-2 text-lg text-brand-text-light">Browse our collection of authentic ingredients.</p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-4 gap-8">
          {/* Sidebar for filtering */}
          <aside className="lg:col-span-1">
            {loading ? <FilterSidebarSkeleton /> : (
              <div className="bg-white p-6 rounded-lg shadow sticky top-24">
                <h3 className="text-xl font-bold font-heading mb-4">Filters</h3>
                
                <div className="mb-6 relative">
                  <label htmlFor="search" className="block text-sm font-medium text-gray-700 mb-1">Search</label>
                  <Input 
                    type="text" 
                    id="search"
                    placeholder="e.g. Fufu flour"
                    value={searchTerm}
                    onChange={handleSearchChange}
                    onFocus={() => searchTerm.length > 1 && setIsSuggestionsVisible(true)}
                    onBlur={() => setTimeout(() => setIsSuggestionsVisible(false), 200)} // delay to allow click
                    autoComplete="off"
                  />
                  {isSuggestionsVisible && suggestions.length > 0 && (
                      <ul className="absolute z-10 w-full bg-white border border-gray-200 rounded-md mt-1 shadow-lg max-h-60 overflow-y-auto">
                          {suggestions.map(product => (
                              <li key={product.id}>
                                  <button
                                      className="w-full text-left px-4 py-2 hover:bg-gray-100 flex items-center gap-4"
                                      onClick={() => handleSuggestionClick(product)}
                                  >
                                      <img src={product.image} alt={product.name} className="w-10 h-10 object-cover rounded" />
                                      <div>
                                          <p className="font-semibold">{product.name}</p>
                                          <p className="text-xs text-gray-500">${product.price.toFixed(2)}</p>
                                      </div>
                                  </button>
                              </li>
                          ))}
                      </ul>
                  )}
                </div>

                <div>
                  <h4 className="font-semibold mb-2">Category</h4>
                  <ul className="space-y-2">
                    {categories.map(category => (
                      <li key={category.id}>
                        <button
                          onClick={() => setSelectedCategory(category.name)}
                          className={`w-full text-left p-2 rounded-md transition-colors ${selectedCategory === category.name ? 'bg-brand-purple/10 text-brand-purple font-bold' : 'hover:bg-gray-100'}`}
                        >
                          {category.name} 
                          {category.id !== 'all' && <span className="text-sm text-gray-500"> ({category.productCount})</span>}
                        </button>
                      </li>
                    ))}
                  </ul>
                </div>
              </div>
            )}
          </aside>

          {/* Product grid */}
          <main className="lg:col-span-3">
            <div className="flex justify-between items-center mb-6 bg-white p-4 rounded-lg shadow">
            {loading ? (
                <>
                  <Skeleton className="h-5 w-32" />
                  <Skeleton className="h-10 w-48" />
                </>
              ) : (
                <>
                  <p className="text-sm text-gray-600">Showing {filteredAndSortedProducts.length} products</p>
                  <div>
                    <label htmlFor="sort" className="sr-only">Sort by</label>
                    <Select id="sort" value={sortOrder} onChange={e => setSortOrder(e.target.value)}>
                      <SelectValue value="name-asc">Sort by name (A-Z)</SelectValue>
                      <SelectValue value="name-desc">Sort by name (Z-A)</SelectValue>
                      <SelectValue value="price-asc">Sort by price (low to high)</SelectValue>
                      <SelectValue value="price-desc">Sort by price (high to low)</SelectValue>
                    </Select>
                  </div>
                </>
              )}
            </div>

            {loading ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-8">
                {Array.from({ length: 9 }).map((_, index) => <ProductCardSkeleton key={index} />)}
              </div>
            ) : filteredAndSortedProducts.length > 0 ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-8">
                {filteredAndSortedProducts.map(product => (
                  <ProductCard key={product.id} product={product} />
                ))}
              </div>
            ) : (
              <div className="text-center py-20 bg-white rounded-lg shadow">
                <h3 className="text-xl font-bold">No Products Found</h3>
                <p className="mt-2 text-gray-600">Try adjusting your filters.</p>
              </div>
            )}
          </main>
        </div>
      </div>
    </div>
  );
};

export default ShopPage;